/* Copyright (c) 2017-2023 VMware, Inc. All rights reserved. -- VMware Confidential */
import {FileMigrator, SubContentTransformation} from "./fileMigrator";
import {TransformationInfo} from "../transformationInfo";

export class HtmlFileMigrator extends FileMigrator {
   private static FAKE_PATH = "C://hack.html";

   constructor(filePath: string, fileContent: string) {
      super(filePath, fileContent);
   }

   migrateFile(transformationInfo: TransformationInfo) {
      const scriptContents: string[] = this.getScriptsTagContent();

      scriptContents.forEach((scriptContent) => {
         const file: FileMigrator = new FileMigrator(HtmlFileMigrator.FAKE_PATH, scriptContent);
         file.migrateFile(transformationInfo);
         this.setContent(this.getContent().replace(scriptContent, file.getContent()));
      });

      const onClickEvents: string[] = this.getOnClickEventsContent();

      onClickEvents.forEach((onClick) => {
         const file: FileMigrator = new FileMigrator(HtmlFileMigrator.FAKE_PATH, onClick);
         file.migrateFile(transformationInfo);
         this.setContent(this.getContent().replace(onClick, file.getContent()));
      });
   }

   getAllFileTransformations(transformationInfo: TransformationInfo): SubContentTransformation[] {
      const scriptContents: string[] = this.getScriptsTagContent();

      let result: SubContentTransformation[] = [];
      let indexOfLastModification: number = 0;
      scriptContents.forEach((scriptContent) => {
         indexOfLastModification = this.getContent()
            .indexOf(scriptContent, indexOfLastModification);

         const fileTrans: SubContentTransformation[] =
            this.getFileTransformations(scriptContent, transformationInfo);

         fileTrans.forEach((trans) => {
            trans.position += indexOfLastModification;
            trans.line += HtmlFileMigrator.countNewLines(
               this.getContent().substr(0, indexOfLastModification));
         });
         result = result.concat(fileTrans);
      });
      result = result.concat(this.parseOnClickEvents(transformationInfo));
      return result;
   }

   private getScriptsTagContent() {
      const regex = /<script[\s\S]*?>([\s\S]*?)<\/script[\s\S]*?>/g;
      return this.extractAllMatchesByGroupIndex(regex, 1);
   }

   private getOnClickEventsContent(): string[] {
      const regex = /\s+onclick\s*=\s*(["'])([\s\S]*?)\1/ig;
      return this.extractAllMatchesByGroupIndex(regex, 2);
   }

   private extractAllMatchesByGroupIndex(regex: RegExp, index: number) : string[] {
      let match;
      let result = [];
      while ((match = regex.exec(this.getContent())) !== null) {
         // This is necessary to avoid infinite loops with zero-width matches
         if (match.index === regex.lastIndex) {
            regex.lastIndex++;
         }
         result.push(match[index])
      }

      return result;
   }

   private getFileTransformations(content: string, transformationInfo: TransformationInfo): SubContentTransformation[] {
      const file: FileMigrator = new FileMigrator(HtmlFileMigrator.FAKE_PATH, content);
      return file.getAllFileTransformations(transformationInfo);
   }

   private parseOnClickEvents(transformationInfo: TransformationInfo): SubContentTransformation[] {
      const onClickEvents: string[] = this.getOnClickEventsContent();

      let result: SubContentTransformation[] = [];
      let indexOfLastModification: number = 0;
      onClickEvents.forEach((onClick) => {
         indexOfLastModification = this.getContent().indexOf(onClick, indexOfLastModification);

         const fileTrans: SubContentTransformation[] =
            this.getFileTransformations(onClick, transformationInfo);

         const contentPrefix: string = this.getContent().substr(0, indexOfLastModification);
         fileTrans.forEach((trans) => {
            trans.position += indexOfLastModification
            trans.line += HtmlFileMigrator.countNewLines(contentPrefix);
            trans.character += HtmlFileMigrator.countCharactersFromLastNewLine(contentPrefix);
         });
         result = result.concat(fileTrans);
      });

      return result;
   }
}
